<?php
/**
 * Plugin Name: Tawhub WhatsApp Integration
 * Description: Complete WooCommerce WhatsApp integration. Send automatic notifications for any order status, notify admins of new orders, and use custom placeholders for rich messages.
 * Version: 2.0
 * Author: Tawhub
 * Author URI: https://tawhub.com
 */
if (! defined('ABSPATH')) {
    exit;
}

// ------------------------------------------------------------------------
// 1. Settings Registration
// ------------------------------------------------------------------------

function tawhub_settings_init()
{
    register_setting('tawhubPlugin', 'tawhub_appkey');
    register_setting('tawhubPlugin', 'tawhub_authkey');
    register_setting('tawhubPlugin', 'tawhub_admin_phone');
    register_setting('tawhubPlugin', 'tawhub_notify_admin_new_order');
    register_setting('tawhubPlugin', 'tawhub_admin_new_order_template', ['default' => "🔔 *New Order Received!*\n\nOrder #{order_id}\nTotal: {order_total}\nCustomer: {billing_first_name} {billing_last_name}\n\nItems:\n{order_items}"]);

    // Register settings for each WooCommerce order status
    $statuses = wc_get_order_statuses();
    foreach ($statuses as $status_key => $status_name) {
        // status_key is like 'wc-pending', we want just 'pending'
        $status_slug = substr($status_key, 3);
        register_setting('tawhubPlugin', "tawhub_notify_customer_{$status_slug}");
        register_setting('tawhubPlugin', "tawhub_customer_{$status_slug}_template", ['default' => "Mesaage for status {$status_name}:\nOrder #{order_id} is now {$status_name}."]);
    }
}
add_action('admin_init', 'tawhub_settings_init');

function tawhub_add_admin_menu()
{
    add_options_page('Tawhub WhatsApp', 'Tawhub WhatsApp', 'manage_options', 'tawhub-whatsapp', 'tawhub_options_page');
}
add_action('admin_menu', 'tawhub_add_admin_menu');

// ------------------------------------------------------------------------
// 2. Settings Page UI
// ------------------------------------------------------------------------

function tawhub_options_page()
{
    // Check if WC is active
    if (! class_exists('WooCommerce')) {
        echo '<div class="error"><p>WooCommerce is not active. Please install WooCommerce to use this plugin.</p></div>';

        return;
    }

    $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'general';
    ?>
    <div class="wrap">
        <h1>Tawhub WhatsApp Integration</h1>
        
        <h2 class="nav-tab-wrapper">
            <a href="?page=tawhub-whatsapp&tab=general" class="nav-tab <?php echo $active_tab == 'general' ? 'nav-tab-active' : ''; ?>">General Settings</a>
            <a href="?page=tawhub-whatsapp&tab=admin" class="nav-tab <?php echo $active_tab == 'admin' ? 'nav-tab-active' : ''; ?>">Admin Notifications</a>
            <a href="?page=tawhub-whatsapp&tab=customer" class="nav-tab <?php echo $active_tab == 'customer' ? 'nav-tab-active' : ''; ?>">Customer Notifications</a>
            <a href="?page=tawhub-whatsapp&tab=test" class="nav-tab <?php echo $active_tab == 'test' ? 'nav-tab-active' : ''; ?>">Test Message</a>
        </h2>

        <form action="options.php" method="post">
            <?php
            settings_fields('tawhubPlugin');

    if ($active_tab == 'general') {
        ?>
                <h3>API Configuration</h3>
                <p>Get your keys from <a href="https://tawhub.com/user/apps" target="_blank">Tawhub Dashboard</a>.</p>
                <table class="form-table">
                    <tr>
                        <th scope="row">App Key</th>
                        <td><input type="text" name="tawhub_appkey" value="<?php echo esc_attr(get_option('tawhub_appkey')); ?>" class="regular-text" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Auth Key</th>
                        <td><input type="text" name="tawhub_authkey" value="<?php echo esc_attr(get_option('tawhub_authkey')); ?>" class="regular-text" /></td>
                    </tr>
                </table>
                <?php
    } elseif ($active_tab == 'admin') {
        ?>
                <h3>Admin Notifications</h3>
                <table class="form-table">
                    <tr>
                        <th scope="row">Admin Phone Number</th>
                        <td>
                            <input type="text" name="tawhub_admin_phone" value="<?php echo esc_attr(get_option('tawhub_admin_phone')); ?>" class="regular-text" />
                            <p class="description">Includes country code (e.g. 2010xxxxxxxx)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">New Order Notification</th>
                        <td>
                            <label><input type="checkbox" name="tawhub_notify_admin_new_order" value="1" <?php checked(1, get_option('tawhub_notify_admin_new_order'), true); ?> /> Send me a WhatsApp message when a new order is placed</label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Message Template</th>
                        <td>
                            <textarea name="tawhub_admin_new_order_template" rows="8" cols="50" class="large-text"><?php echo esc_attr(get_option('tawhub_admin_new_order_template')); ?></textarea>
                            <p class="description">Available Placeholders: <code>{order_id}, {amount}, {order_total}, {billing_first_name}, {billing_last_name}, {billing_phone}, {status}, {order_items}, {order_date}</code></p>
                        </td>
                    </tr>
                </table>
                <?php
    } elseif ($active_tab == 'customer') {
        $wc_statuses = wc_get_order_statuses();
        ?>
                <h3>Customer Notifications</h3>
                <p>Configure automatic messages sent to customers when their order status changes.</p>
                <p class="description">Available Placeholders: <code>{order_id}, {amount}, {order_total}, {billing_first_name}, {billing_last_name}, {billing_phone}, {status}, {order_items}, {order_date}</code></p>
                <hr>
                <?php
        foreach ($wc_statuses as $status_key => $status_name) {
            $slug = substr($status_key, 3);
            $option_enable = "tawhub_notify_customer_{$slug}";
            $option_template = "tawhub_customer_{$slug}_template";
            ?>
                    <h4><?php echo esc_html($status_name); ?> (<?php echo $slug; ?>)</h4>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Enable</th>
                            <td><label><input type="checkbox" name="<?php echo $option_enable; ?>" value="1" <?php checked(1, get_option($option_enable), true); ?> /> Send message when status is <?php echo $status_name; ?></label></td>
                        </tr>
                        <tr>
                            <th scope="row">Message</th>
                            <td><textarea name="<?php echo $option_template; ?>" rows="4" class="large-text"><?php echo esc_attr(get_option($option_template)); ?></textarea></td>
                        </tr>
                    </table>
                    <hr>
                    <?php
        }
    } elseif ($active_tab == 'test') {
        // Handle Test Message Logic purely here via a separate form handling
        ?>
                 <h3>Send Test Message</h3>
                 <p>Save your API keys in the "General" tab first!</p>
                 <div class="card" style="max-width: 500px; padding: 1rem; background: #fff; border: 1px solid #ccd0d4;">
                     <label><strong>Phone Number (with country code):</strong></label><br>
                     <input type="text" id="tawhub_test_phone" class="regular-text" placeholder="2010xxxxxxxx" value="<?php echo esc_attr(get_option('tawhub_admin_phone')); ?>"><br><br>
                     
                     <label><strong>Message:</strong></label><br>
                     <textarea id="tawhub_test_message" class="large-text" rows="3">This is a test message from Tawhub Plugin.</textarea><br><br>
                     
                     <button type="button" class="button button-primary" id="tawhub_send_test_btn">Send Test</button>
                     <div id="tawhub_test_result" style="margin-top: 10px;"></div>
                 </div>

                 <script>
                 jQuery(document).ready(function($) {
                     $('#tawhub_send_test_btn').click(function() {
                         var btn = $(this);
                         var phone = $('#tawhub_test_phone').val();
                         var msg = $('#tawhub_test_message').val();
                         
                         btn.prop('disabled', true).text('Sending...');
                         $('#tawhub_test_result').html('');

                         $.post(ajaxurl, {
                             action: 'tawhub_test_message',
                             phone: phone,
                             message: msg
                         }, function(response) {
                             btn.prop('disabled', false).text('Send Test');
                             if(response.success) {
                                 $('#tawhub_test_result').html('<span style="color: green;">' + response.data + '</span>');
                             } else {
                                 $('#tawhub_test_result').html('<span style="color: red;">Error: ' + response.data + '</span>');
                             }
                         });
                     });
                 });
                 </script>
                 <?php
    }

    if ($active_tab != 'test') {
        submit_button();
    }
    ?>
        </form>
    </div>
    <?php
}

// ------------------------------------------------------------------------
// 3. Helper: Parse Orders & Placeholders
// ------------------------------------------------------------------------

function tawhub_get_order_items_list($order)
{
    $items_list = '';
    foreach ($order->get_items() as $item_id => $item) {
        $product_name = $item->get_name();
        $quantity = $item->get_quantity();
        $total = wc_price($item->get_total());
        $items_list .= "- {$product_name} x {$quantity} ({$total})\n";
    }

    return strip_tags($items_list);
}

function tawhub_replace_placeholders($message, $order)
{
    $order_data = $order->get_data();

    $placeholders = [
        '{order_id}' => $order->get_id(),
        '{amount}' => $order->get_total(), // Raw amount
        '{order_total}' => strip_tags($order->get_formatted_order_total()), // Formatted
        '{billing_first_name}' => $order->get_billing_first_name(),
        '{billing_last_name}' => $order->get_billing_last_name(),
        '{billing_phone}' => $order->get_billing_phone(),
        '{status}' => wc_get_order_status_name($order->get_status()),
        '{order_date}' => $order->get_date_created()->date('Y-m-d H:i'),
        '{order_items}' => tawhub_get_order_items_list($order),
    ];

    foreach ($placeholders as $key => $value) {
        $message = str_replace($key, $value, $message);
    }

    return $message;
}

// ------------------------------------------------------------------------
// 4. Core Sending Function
// ------------------------------------------------------------------------

function tawhub_send_whatsapp($to, $message)
{
    $appkey = get_option('tawhub_appkey');
    $authkey = get_option('tawhub_authkey');

    if (empty($appkey) || empty($authkey) || empty($to)) {
        return false;
    }

    // Clean phone number
    $to = preg_replace('/[^0-9]/', '', $to);

    $body = [
        'appkey' => $appkey,
        'authkey' => $authkey,
        'to' => $to,
        'message' => $message,
        'sandbox' => 'false',
    ];

    $response = wp_remote_post('https://tawhub.com/api/create-message', [
        'method' => 'POST',
        'body' => $body,
        'timeout' => 45,
        'blocking' => true,
    ]);

    if (is_wp_error($response)) {
        error_log('Tawhub API Error: '.$response->get_error_message());

        return false;
    }

    return true;
}

// ------------------------------------------------------------------------
// 5. Hooks: Order Status Changed & New Order based
// ------------------------------------------------------------------------

// A. Customer Notifications (Status Changes)
add_action('woocommerce_order_status_changed', 'tawhub_on_order_status_change', 10, 4);
function tawhub_on_order_status_change($order_id, $old_status, $new_status, $order)
{
    // Check if enabled for this new status
    $option_name = "tawhub_notify_customer_{$new_status}";
    if (! get_option($option_name)) {
        return;
    }

    // Get number
    $phone = $order->get_billing_phone();
    if (empty($phone)) {
        return;
    }

    // Get template
    $template = get_option("tawhub_customer_{$new_status}_template");
    if (empty($template)) {
        return;
    }

    // Parse
    $message = tawhub_replace_placeholders($template, $order);

    // Send
    tawhub_send_whatsapp($phone, $message);
}

// B. Admin Notification (New Order)
add_action('woocommerce_new_order', 'tawhub_on_new_order', 10, 1);
function tawhub_on_new_order($order_id)
{
    if (! get_option('tawhub_notify_admin_new_order')) {
        return;
    }

    $admin_phone = get_option('tawhub_admin_phone');
    if (empty($admin_phone)) {
        return;
    }

    $order = wc_get_order($order_id);
    if (! $order) {
        return;
    }

    $template = get_option('tawhub_admin_new_order_template');
    if (empty($template)) {
        return;
    }

    $message = tawhub_replace_placeholders($template, $order);

    tawhub_send_whatsapp($admin_phone, $message);
}

// ------------------------------------------------------------------------
// 6. Test Message AJAX
// ------------------------------------------------------------------------
add_action('wp_ajax_tawhub_test_message', 'tawhub_test_message_callback');
function tawhub_test_message_callback()
{
    if (! current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }

    $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
    $message = isset($_POST['message']) ? sanitize_textarea_field($_POST['message']) : '';

    if (tawhub_send_whatsapp($phone, $message)) {
        wp_send_json_success('Message sent successfully!');
    } else {
        wp_send_json_error('Failed to send. Check logs or API Keys.');
    }
}
